/*
 * Decompiled with CFR 0.152.
 */
package com.nuix.automate.workflow.core.utils.digestlist;

import com.nuix.automate.utils.general.FileUtils;
import com.nuix.automate.utils.logging.LogManagerUtils;
import com.nuix.automate.utils.logging.LoggerWrapper;
import com.nuix.automate.workflow.core.utils.digestlist.DigestHelperUtils;
import com.nuix.automate.workflow.core.utils.digestlist.DigestLocation;
import com.nuix.automate.workflow.core.utils.digestlist.DigestOperation;
import com.nuix.automate.workflow.core.utils.digestlist.MD5Hash;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.EOFException;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import javax.xml.bind.DatatypeConverter;
import nuix.Case;

public class DigestHelper {
    private static final LoggerWrapper LOGGER = LogManagerUtils.getLogger(DigestHelper.class);
    private static final byte[] FILE_HEADER = new byte[]{70, 50, 68, 76, 0, 0, 0, 1, 0, 3, 77, 68, 53};
    private Map<String, Path> listNameToPath = new HashMap<String, Path>();
    private Map<DigestLocation, Path> directoryLocationToPath = new LinkedHashMap<DigestLocation, Path>(3);
    private long itemsProcessed;
    private long itemsToProcess;
    private boolean stopRequested;

    public DigestHelper(Case nuixCase) throws IOException {
        this.resetState(1L);
        this.populateMaps(nuixCase);
    }

    public void createDigestList(String listName, DigestLocation location, Set<MD5Hash> md5SetToAdd) throws IOException {
        this.importDigestList(listName, location, md5SetToAdd, false);
    }

    public int addToDigestList(String listName, Set<MD5Hash> md5SetToAdd) throws IOException {
        Path digestList = this.getPathFromMap(listName);
        HashSet<MD5Hash> copyOfMd5SetToAdd = new HashSet<MD5Hash>(md5SetToAdd);
        this.resetState(copyOfMd5SetToAdd.size());
        try (BufferedInputStream bufferedInputStream = new BufferedInputStream(Files.newInputStream(digestList, new OpenOption[0]));){
            if (bufferedInputStream.read(new byte[13], 0, 13) == -1) {
                throw new EOFException("Unsupported file format");
            }
            byte[] buffer = new byte[16];
            MD5Hash md5Hash = new MD5Hash(buffer);
            while (bufferedInputStream.read(buffer) > 0) {
                if (copyOfMd5SetToAdd.remove(md5Hash)) {
                    ++this.itemsProcessed;
                }
                if (!this.stopRequested && !copyOfMd5SetToAdd.isEmpty()) continue;
                int n = 0;
                return n;
            }
        }
        try (BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(Files.newOutputStream(digestList, StandardOpenOption.APPEND));){
            int appendedCount = 0;
            for (MD5Hash md5Hash : copyOfMd5SetToAdd) {
                if (this.stopRequested) {
                    int n = appendedCount;
                    return n;
                }
                byte[] md5ByteArray = md5Hash.getByteArray();
                bufferedOutputStream.write(md5ByteArray);
                ++this.itemsProcessed;
                ++appendedCount;
            }
        }
        return copyOfMd5SetToAdd.size();
    }

    public int removeFromDigestList(String listName, Set<MD5Hash> md5SetToRemove) throws IOException {
        Path digestList = this.getPathFromMap(listName);
        Path tempDigestList = FileUtils.createTempFilePath((String)listName, (String)".hash", (FileAttribute[])new FileAttribute[0]);
        int removedCount = 0;
        this.resetState(Files.size(digestList) / 16L);
        try (BufferedInputStream digestListInputStream = new BufferedInputStream(Files.newInputStream(digestList, new OpenOption[0]));
             BufferedOutputStream tempDigestListOutputStream = new BufferedOutputStream(Files.newOutputStream(tempDigestList, new OpenOption[0]));){
            if (digestListInputStream.read(new byte[13], 0, 13) == -1) {
                throw new EOFException("Unsupported file format");
            }
            tempDigestListOutputStream.write(FILE_HEADER);
            byte[] buffer = new byte[16];
            MD5Hash md5Hash = new MD5Hash(buffer);
            while (digestListInputStream.read(buffer) > 0) {
                if (this.stopRequested) {
                    break;
                }
                if (md5SetToRemove.contains(md5Hash)) {
                    ++removedCount;
                } else {
                    tempDigestListOutputStream.write(buffer);
                }
                ++this.itemsProcessed;
            }
        }
        Files.move(tempDigestList, digestList, StandardCopyOption.REPLACE_EXISTING);
        return removedCount;
    }

    public boolean getDigestListExists(String digestListName, DigestLocation resultLocation) {
        Path resultList = this.getPathFromLocation(digestListName, resultLocation);
        return Files.exists(resultList, new LinkOption[0]);
    }

    public int manageDigestLists(String listAName, String listBName, DigestOperation operation, String resultName, DigestLocation resultLocation, boolean overwriteIfExists) throws IOException {
        Path digestListA = this.getPathFromMap(listAName);
        Path digestListB = this.getPathFromMap(listBName);
        Path resultList = this.getPathFromLocation(resultName, resultLocation);
        if (Files.exists(resultList, new LinkOption[0]) && !overwriteIfExists) {
            throw new FileAlreadyExistsException("Resulting digest list already exists");
        }
        DigestHelperUtils.resetState();
        Set<MD5Hash> operateOnMd5s = DigestHelperUtils.getMd5HashSetFromFile(digestListA);
        Set<MD5Hash> operateWithMd5s = DigestHelperUtils.getMd5HashSetFromFile(digestListB);
        switch (operation) {
            case ADD: {
                operateOnMd5s.addAll(operateWithMd5s);
                this.writeDigestList(resultList, operateOnMd5s);
                return operateOnMd5s.size();
            }
            case SUBTRACT: {
                operateOnMd5s.removeAll(operateWithMd5s);
                this.writeDigestList(resultList, operateOnMd5s);
                return operateOnMd5s.size();
            }
            case INTERSECT: {
                operateOnMd5s.retainAll(operateWithMd5s);
                this.writeDigestList(resultList, operateOnMd5s);
                return operateOnMd5s.size();
            }
        }
        throw new IllegalArgumentException("Given operation is not supported");
    }

    public void importDigestList(String listName, DigestLocation location, Set<MD5Hash> md5SetToImport, boolean overwriteIfExists) throws IOException {
        Path digestList = this.getPathFromLocation(listName, location);
        if (Files.exists(digestList, new LinkOption[0]) && !overwriteIfExists) {
            throw new FileAlreadyExistsException("Digest list already exists");
        }
        this.writeDigestList(digestList, md5SetToImport);
    }

    public void exportDigestList(String listName, String exportFile, boolean overwriteIfExists) throws IOException {
        Path digestList = this.getPathFromMap(listName);
        Path exportPath = Paths.get(exportFile, new String[0]);
        if (Files.exists(exportPath, new LinkOption[0]) && !overwriteIfExists) {
            throw new FileAlreadyExistsException("File with same name already exists");
        }
        Path exportParentDirectory = exportPath.getParent();
        if (!Files.exists(exportParentDirectory, new LinkOption[0])) {
            Files.createDirectories(exportParentDirectory, new FileAttribute[0]);
        }
        this.resetState(Files.size(digestList) / 16L);
        try (BufferedInputStream inputStream = new BufferedInputStream(Files.newInputStream(digestList, new OpenOption[0]));
             BufferedWriter outputStream = Files.newBufferedWriter(exportPath, new OpenOption[0]);){
            if (inputStream.read(new byte[13], 0, 13) == -1) {
                throw new EOFException("Unsupported file format");
            }
            byte[] md5ByteArray = new byte[16];
            while (inputStream.read(md5ByteArray) > 0) {
                if (this.stopRequested) {
                    break;
                }
                String md5Hex = DatatypeConverter.printHexBinary((byte[])md5ByteArray).toLowerCase();
                outputStream.write(md5Hex);
                outputStream.newLine();
                ++this.itemsProcessed;
            }
        }
    }

    public void deleteDigestList(String listName, Set<DigestLocation> locations) throws IOException {
        this.resetState(locations.size());
        for (DigestLocation location : locations) {
            if (this.stopRequested) break;
            Path digestList = this.getPathFromLocation(listName, location);
            FileUtils.deleteRecursively((Path)digestList);
            ++this.itemsProcessed;
        }
    }

    public double getExecutionProgress() {
        return (double)this.itemsProcessed / (double)this.itemsToProcess;
    }

    public long getItemsProcessed() {
        return this.itemsProcessed;
    }

    public void requestStop() {
        this.stopRequested = true;
        DigestHelperUtils.requestStop();
    }

    private void writeDigestList(Path digestList, Set<MD5Hash> md5HashSet) throws IOException {
        try (BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(Files.newOutputStream(digestList, new OpenOption[0]));){
            this.resetState(md5HashSet.size());
            bufferedOutputStream.write(FILE_HEADER);
            for (MD5Hash md5Hash : md5HashSet) {
                if (this.stopRequested) {
                    break;
                }
                byte[] md5ByteArray = md5Hash.getByteArray();
                bufferedOutputStream.write(md5ByteArray);
                ++this.itemsProcessed;
            }
        }
    }

    private Path getPathFromMap(String listName) throws NoSuchFileException {
        Path digestList = this.listNameToPath.get(listName);
        if (digestList == null || !Files.exists(digestList, new LinkOption[0])) {
            throw new NoSuchFileException("Digest list " + listName + " could not be found");
        }
        return digestList;
    }

    private Path getPathFromLocation(String listName, DigestLocation location) {
        Path digestDir = this.directoryLocationToPath.get((Object)location);
        if (digestDir == null) {
            throw new IllegalArgumentException("Given digest location doesn't exist");
        }
        return digestDir.resolve(listName + ".hash");
    }

    private void populateMaps(Case nuixCase) throws IOException {
        if (System.getProperty("os.name").contains("Windows")) {
            this.directoryLocationToPath.put(DigestLocation.LOCAL_COMPUTER, Paths.get(System.getenv("PROGRAMDATA"), "Nuix\\Digest Lists"));
            this.directoryLocationToPath.put(DigestLocation.USER, Paths.get(System.getenv("APPDATA"), "Nuix\\Digest Lists"));
        } else if (System.getProperty("user.home") != null) {
            this.directoryLocationToPath.put(DigestLocation.USER, Paths.get(System.getProperty("user.home"), ".nuix/Digest Lists"));
        }
        if (nuixCase != null) {
            this.directoryLocationToPath.put(DigestLocation.CASE, nuixCase.getLocation().toPath().resolve("Stores/User Data/Digest Lists"));
        }
        for (Path digestListDir : this.directoryLocationToPath.values()) {
            if (!Files.exists(digestListDir, new LinkOption[0])) {
                Files.createDirectories(digestListDir, new FileAttribute[0]);
            }
            DirectoryStream<Path> pathStream = Files.newDirectoryStream(digestListDir);
            try {
                for (Path path : pathStream) {
                    String fileName = path.getFileName().toString();
                    if (!fileName.endsWith(".hash")) continue;
                    String listName = fileName.replace(".hash", "");
                    this.listNameToPath.put(listName, path);
                }
            }
            finally {
                if (pathStream == null) continue;
                pathStream.close();
            }
        }
    }

    private void resetState(long total) {
        this.itemsToProcess = total;
        this.itemsProcessed = 0L;
        this.stopRequested = false;
        DigestHelperUtils.resetState();
    }
}

