/*
 * Decompiled with CFR 0.152.
 */
package com.nuix.graph.playbook.loader;

import com.google.common.collect.ImmutableList;
import com.nuix.graph.playbook.Settings;
import com.nuix.graph.playbook.engine.entities.Edge;
import com.nuix.graph.playbook.engine.entities.Node;
import com.nuix.graph.playbook.engine.entities.SourceNode;
import com.nuix.graph.playbook.loader.MemGraphStore;
import com.nuix.graph.playbook.loader.entities.GraphObject;
import com.nuix.graph.playbook.loader.entities.StoredEdge;
import com.nuix.graph.playbook.loader.entities.StoredNode;
import java.util.List;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;
import org.neo4j.driver.Session;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MemGraphLoader
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(MemGraphLoader.class);
    private final BlockingDeque<List<GraphObject>> pendingObjects;
    private final Session session;
    private final List<String> runBeforeLoad;
    private final List<String> runAfterLoad;
    private final MemGraphStore store;
    private final Settings settings;
    private final Thread loadingThread;
    private boolean closed;

    public MemGraphLoader(Session session, Settings settings) {
        this(session, (List<String>)ImmutableList.of(), (List<String>)ImmutableList.of(), new MemGraphStore(session, settings), settings);
    }

    public MemGraphLoader(Session session, List<String> runBeforeLoad, List<String> runAfterLoad, MemGraphStore memGraphStore, Settings settings) {
        this.session = session;
        this.runBeforeLoad = runBeforeLoad;
        this.runAfterLoad = runAfterLoad;
        this.pendingObjects = new LinkedBlockingDeque<List<GraphObject>>(settings.getLoaderQueueSize());
        this.store = memGraphStore;
        this.settings = settings;
        if (!runBeforeLoad.isEmpty()) {
            if (log.isDebugEnabled()) {
                log.debug("Running operations before load:\n{}", (Object)runBeforeLoad);
            }
            runBeforeLoad.forEach(arg_0 -> ((Session)session).run(arg_0));
        }
        this.loadingThread = new Thread(this::loadingLoop, settings.getGraphLoaderThreadName());
        this.loadingThread.start();
    }

    public void loadNodes(List<SourceNode> sourceNodes) {
        if (this.closed) {
            throw new IllegalStateException("Loader thread is already closed and can't accept new requests");
        }
        if (sourceNodes.isEmpty()) {
            return;
        }
        ImmutableList.Builder builder = ImmutableList.builder();
        for (SourceNode source : sourceNodes) {
            builder.add((Object)StoredNode.from(source));
            for (Edge edge : source.getEdges()) {
                Node target = edge.getTarget();
                builder.add((Object)StoredNode.from(target));
                builder.add((Object)StoredEdge.from(source, edge));
            }
        }
        this.pendingObjects.put((List<GraphObject>)builder.build());
    }

    public long getQueueSize() {
        return this.pendingObjects.size();
    }

    public long getCacheSize() {
        return this.store.getCacheSize();
    }

    public int getPendingEdgesCount() {
        return this.store.getPendingEdgesCount();
    }

    public int getPendingNodesCount() {
        return this.store.getPendingNodesCount();
    }

    public boolean isLoading() {
        return this.loadingThread.isAlive();
    }

    public int getTotalNodesCommitted() {
        return this.store.getTotalNodesCommitted();
    }

    public int getTotalEdgesCommitted() {
        return this.store.getTotalEdgesCommitted();
    }

    public void abort() {
        log.warn("Aborting loading thread.");
        this.stopLoadingThread();
    }

    private void stopLoadingThread() {
        log.debug("Stopping loading thread");
        try {
            this.closed = true;
            this.loadingThread.join(60000L);
            if (this.loadingThread.isAlive()) {
                this.loadingThread.interrupt();
            }
        }
        catch (InterruptedException e) {
            log.error("Interrupted while waiting for loading thread to finish", e);
        }
    }

    @Override
    public void close() {
        if (this.closed) {
            return;
        }
        int ticker = 0;
        while (this.getQueueSize() > 0L) {
            if (ticker++ % 10000 == 0) {
                log.info("Waiting for loading thread to finish. Current queue size: {}", (Object)this.pendingObjects.size());
            }
            try {
                TimeUnit.MILLISECONDS.sleep(100L);
            }
            catch (InterruptedException e) {
                log.error("Interrupted while waiting for loading thread to finish", e);
            }
        }
        this.stopLoadingThread();
        log.info("Flushing store");
        this.store.close();
        if (!this.runAfterLoad.isEmpty()) {
            if (log.isDebugEnabled()) {
                log.debug("Running operations after load:\n{}", (Object)this.runAfterLoad);
            }
            this.runAfterLoad.forEach(arg_0 -> ((Session)this.session).run(arg_0));
        }
    }

    private void loadingLoop() {
        log.debug("Loading loop started.");
        int numberOfObjects = 0;
        int repeatedErrors = 0;
        while (!this.closed) {
            List<GraphObject> objects = this.pendingObjects.poll();
            try {
                if (objects == null) {
                    if (this.store.getPendingNodesCount() + this.store.getPendingEdgesCount() > 0) {
                        this.store.flush();
                    }
                    try {
                        TimeUnit.MILLISECONDS.sleep(100L);
                        continue;
                    }
                    catch (InterruptedException e) {
                        throw new RuntimeException(e);
                    }
                }
                this.store.push(objects);
                if (this.settings.getTransactionSize() <= 0 || (numberOfObjects += objects.size()) >= this.settings.getTransactionSize()) {
                    this.store.flush();
                    numberOfObjects = 0;
                }
                repeatedErrors = 0;
            }
            catch (Throwable t) {
                if (++repeatedErrors > 5) {
                    log.error("It looks like we are in an error loop. Cancelling load.", t);
                    this.closed = true;
                    throw t;
                }
                if (objects == null) {
                    log.error("Unexpected error in the loading thread", t);
                    continue;
                }
                log.error("{} objects could not be pushed to MemGraph due to an unexpected exception. Labels and IDs follow.", (Object)objects.size(), (Object)t);
                for (GraphObject object : objects) {
                    if (object instanceof StoredNode) {
                        log.error("Couldn't load: {}:{}", (Object)object.getLabel(), ((StoredNode)object).getId());
                        continue;
                    }
                    if (!(object instanceof StoredEdge)) continue;
                    StoredEdge e = (StoredEdge)object;
                    log.error("Couldn't load: {}:{}={}=>{}:{}", e.getSourceLabel(), e.getSourceId(), e.getLabel(), e.getTargetLabel(), e.getTargetId());
                }
            }
        }
        this.closed = true;
        log.info("Leaving the loading loop");
    }
}

