
const colors = {
    white: 'white',
    red: 'red',
    yellow: 'yellow',
    orange: 'orange',
    green: 'green',
    blue: 'blue',
    workflowBlue: 'rgb(0, 86, 150)',
    purple: 'purple',
    brown: 'brown',
    black: 'black'
}

const backgroundColors = {
    transparent: 'transparent',
    ...colors
}

function workerScript() {
    function buildStyle(highlightOption) {
        const {color, backgroundColor, isBold, isItalic, isOverline, isUnderline, isLineThrough, lineColor, lineStyle} = highlightOption;

        const style = {
            color: colors[color],
            backgroundColor: backgroundColors[backgroundColor]
        };

        if (isBold)
            style.fontWeight = 600;
        if (isItalic)
            style.fontStyle = 'italic';
        if (isOverline || isUnderline || isLineThrough) {
            const decorations = [
                isOverline && 'overline',
                isUnderline && 'underline',
                isLineThrough && 'line-through',
                colors[lineColor],
                lineStyle
            ];
            // text-decoration: overline grey wavy
            style.textDecoration = decorations.filter(_d => _d).join(' ');
        }

        return style;
    }

    function applyHighlightToArray(options) {
        const {key, arr, regex, style} = options;
        const res = [];

        for (let i = 0; i < arr.length; i++) {
            if (typeof arr[i] === 'string') {
                const result = applyHighlight({key: `${key}_TEXT[${i}]`, text: arr[i], regex, style});
                for (const e of result) {
                    res.push(e);
                }
            } else {
                res.push(arr[i]);
            }
        }

        return res;
    }

    function applyHighlight(options) {
        const {key, text, regex, style} = options;

        const textSplit = [];
        let lastIndex = 0;

        let matches = [];
        try {
            const regExp = new RegExp(regex, 'g');
            matches = text.matchAll(regExp);
        } catch {}

        for (const _match of matches) {
            const {0: value, index} = _match;

            if (!value)
                continue;

            // Add portion of string before current match and after previous match (@lastIndex) to array @textSplit
            textSplit.push(
                text.substring(lastIndex, index)
            );
            // Update @lastIndex to index after match value
            lastIndex = index + value.length;

            // Place match value into array with highlighting JSX/element
            textSplit.push(
                {
                    key: `${key}_MATCH[${index}]`,
                    style,
                    value,
                }
            );
        }

        // Add remainder of text to end of array @textSplit
        // (Adds all text if no matches)
        textSplit.push(
            text.substring(lastIndex)
        );

        return textSplit;
    }

    function getHighlightText(textArr, highlightOptions) {
        let textSplit = Array.isArray(textArr) ? textArr : [textArr];

        const options = Object.keys(highlightOptions)
            .map(key => highlightOptions[key])

            // Filter for enabled + regex defined
            .filter(_option => _option.enabled && _option.regex);

        for (let i = 0; i < options.length; i++) {

            const style = buildStyle(options[i]);
            textSplit = applyHighlightToArray({key: `HIGHLIGHT[${i}]`, arr: textSplit, regex: options[i].regex, style});
        }

        return textSplit;
    }

    workerScript.run = function(text, highlightOptions) {
        let textArr = [text];
        if (Array.isArray(text)) {
            textArr = text;
        }

        return getHighlightText(textArr, highlightOptions);
    }

    onmessage = function (e) {
        if (!Array.isArray(e.data)) {
            return
        }
        const [id, text, highlightOptions, key] = e.data;
        if (key !== 'ramp')
            return;

        postMessage(
            [id, workerScript.run(text, highlightOptions)]
        );
    }
}
workerScript();


let useWebWorker = false;
let worker, scriptUrl;
let workerTimeout = null;
const callbackMap = {};

function startHighlightWorker({text, highlightOptions, callback}) {

    if (!useWebWorker) {
        // Hacking worker postMessage result
        callback(
            workerScript.run(text, highlightOptions)
        );
        return;
    }

    try {
        if (scriptUrl == null) {
            const scriptBlob = new Blob([`(${workerScript.toString()})()`], {type: 'text/javascript'});
            scriptUrl = URL.createObjectURL(scriptBlob);
        }

        if (worker == null) {
            worker = new Worker(scriptUrl);

            worker.onmessage = e => {
                const [id, result] = e.data;

                if (typeof callbackMap[id] === 'function') {
                    callbackMap[id](result);
                    delete callbackMap[id];
                }

                if (workerTimeout != null) {
                    clearTimeout(workerTimeout);
                    workerTimeout = null;
                }
            }
        }

        if (workerTimeout == null) {
            workerTimeout = setTimeout(() => {
                // Reset and disable regexes
                worker.terminate();
                worker = null;
                workerTimeout = null;

                window.location.href = '#/?disableHighlightText';
                window.location.reload();
            }, 5000);
        }

        const id = generateUUID4();
        callbackMap[id] = callback;
        // Using postMessage to call runScript function
        worker.postMessage([id, text, highlightOptions, 'ramp']);
    } catch (error) {
        // If webWorker fails, run manually
        useWebWorker = false;
        startHighlightWorker({text, highlightOptions, callback});
    }
}

const generateUUID4 = function() {
    return ([1e7]+-1e3+-4e3+-8e3+-1e11).replace(/[018]/g, c =>
        (c ^ crypto.getRandomValues(new Uint8Array(1))[0] & 15 >> c / 4).toString(16)
    );
}
