/*
 * Decompiled with CFR 0.152.
 */
package ai.djl.ndarray.internal;

import ai.djl.ndarray.NDArray;
import ai.djl.ndarray.NDScope;
import ai.djl.ndarray.types.DataType;
import ai.djl.ndarray.types.Shape;
import ai.djl.util.Utils;
import java.lang.management.ManagementFactory;
import java.util.Arrays;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public abstract class NDFormat {
    private static final int PRECISION = 8;
    private static final String LF = System.lineSeparator();
    private static final Pattern PATTERN = Pattern.compile("\\s*\\d\\.(\\d*?)0*e[+-](\\d+)");
    private static final boolean DEBUGGER = !Boolean.getBoolean("jshell") && ManagementFactory.getRuntimeMXBean().getInputArguments().stream().anyMatch(arg -> arg.startsWith("-agentlib:jdwp"));

    public static String format(NDArray array, int maxSize, int maxDepth, int maxRows, int maxColumns) {
        return NDFormat.format(array, maxSize, maxDepth, maxRows, maxColumns, !DEBUGGER);
    }

    public static String format(NDArray array, int maxSize, int maxDepth, int maxRows, int maxColumns, boolean withContent) {
        StringBuilder sb = new StringBuilder(1000);
        String name = array.getName();
        if (name != null) {
            sb.append(name).append(": ");
        } else {
            sb.append("ND: ");
        }
        sb.append(array.getShape()).append(' ').append(array.getDevice()).append(' ').append((Object)array.getDataType());
        if (array.hasGradient()) {
            sb.append(" hasGradient");
        }
        if (!withContent) {
            sb.append("\nCheck the \"Development Guideline\"->Debug to enable array display.\n");
            return sb.toString();
        }
        DataType dataType = array.getDataType();
        NDFormat format = dataType == DataType.BOOLEAN ? new BooleanFormat() : (dataType == DataType.STRING ? new StringFormat() : (dataType.isInteger() ? new IntFormat() : new FloatFormat()));
        return format.dump(sb, array, maxSize, maxDepth, maxRows, maxColumns);
    }

    protected abstract CharSequence format(Number var1);

    protected void init(NDArray array) {
    }

    protected String dump(StringBuilder sb, NDArray array, int maxSize, int maxDepth, int maxRows, int maxColumns) {
        sb.append(LF);
        long size = array.size();
        long dimension = array.getShape().dimension();
        if (size == 0L) {
            sb.append("[]").append(LF);
        } else if (dimension == 0L) {
            this.init(array);
            sb.append(this.format(array.toArray()[0])).append(LF);
        } else if (size > (long)maxSize) {
            sb.append("Exceed max print size:").append(LF);
            int limit = Math.min(maxSize, maxRows * maxColumns);
            this.dumpFlat(sb, array, limit);
        } else if (dimension > (long)maxDepth) {
            sb.append("Exceed max print dimension:").append(LF);
            int limit = Math.min(maxSize, maxRows * maxColumns);
            this.dumpFlat(sb, array, limit);
        } else {
            this.init(array);
            this.dump(sb, array, 0, true, maxRows, maxColumns);
        }
        return sb.toString();
    }

    private void dump(StringBuilder sb, NDArray array, int depth, boolean first, int maxRows, int maxColumns) {
        if (!first) {
            Utils.pad(sb, ' ', depth);
        }
        sb.append('[');
        Shape shape = array.getShape();
        if (shape.dimension() == 1) {
            this.append(sb, array.toArray(), maxColumns);
        } else {
            long len = shape.head();
            long limit = Math.min(len, (long)maxRows);
            int i = 0;
            while ((long)i < limit) {
                try (NDArray nd = array.get(i);){
                    this.dump(sb, nd, depth + 1, i == 0, maxRows, maxColumns);
                }
                ++i;
            }
            long remaining = len - limit;
            if (remaining > 0L) {
                Utils.pad(sb, ' ', depth + 1);
                sb.append("... ").append(remaining).append(" more");
            }
            Utils.pad(sb, ' ', depth);
        }
        if (depth == 0) {
            sb.append(']').append(LF);
        } else {
            sb.append("],").append(LF);
        }
    }

    private void dumpFlat(StringBuilder sb, NDArray array, int limit) {
        try (NDScope ignore = new NDScope();){
            NDArray tmp = array.flatten().get(":" + limit, new Object[0]);
            this.init(tmp);
            sb.append('{');
            this.append(sb, array.toArray(), limit);
            sb.append('}').append(LF);
        }
    }

    private void append(StringBuilder sb, Number[] values, int maxColumns) {
        if (values.length == 0) {
            return;
        }
        long limit = Math.min(values.length, maxColumns);
        sb.append(this.format(values[0]));
        int i = 1;
        while ((long)i < limit) {
            sb.append(", ");
            sb.append(this.format(values[i]));
            ++i;
        }
        long remaining = (long)values.length - limit;
        if (remaining > 0L) {
            sb.append(", ... ").append(remaining).append(" more");
        }
    }

    private static final class BooleanFormat
    extends NDFormat {
        private BooleanFormat() {
        }

        @Override
        public CharSequence format(Number value) {
            return value.byteValue() != 0 ? " true" : "false";
        }
    }

    private static final class StringFormat
    extends NDFormat {
        private StringFormat() {
        }

        @Override
        public CharSequence format(Number value) {
            return null;
        }

        @Override
        protected String dump(StringBuilder sb, NDArray array, int maxSize, int maxDepth, int maxRows, int maxColumns) {
            return Arrays.toString(array.toStringArray());
        }
    }

    private static final class IntFormat
    extends NDFormat {
        private boolean exponential;
        private int precision;
        private int totalLength;

        private IntFormat() {
        }

        @Override
        public void init(NDArray array) {
            Number[] values = array.toArray();
            if (values.length == 1) {
                this.totalLength = 1;
                return;
            }
            long max = 0L;
            long negativeMax = 0L;
            for (Number n : values) {
                long v = n.longValue();
                long abs = Math.abs(v);
                if (v < 0L && abs > negativeMax) {
                    negativeMax = abs;
                }
                if (abs <= max) continue;
                max = abs;
            }
            if ((double)max >= 1.0E8) {
                this.exponential = true;
                this.precision = Math.min(8, (int)Math.log10(max) + 1);
            } else {
                int size = max != 0L ? (int)Math.log10(max) + 1 : 1;
                int negativeSize = negativeMax != 0L ? (int)Math.log10(negativeMax) + 2 : 2;
                this.totalLength = Math.max(size, negativeSize);
            }
        }

        @Override
        public CharSequence format(Number value) {
            if (this.exponential) {
                return String.format(Locale.ENGLISH, "% ." + this.precision + "e", Float.valueOf(value.floatValue()));
            }
            return String.format(Locale.ENGLISH, "%" + this.totalLength + "d", value.longValue());
        }
    }

    private static final class FloatFormat
    extends NDFormat {
        private boolean exponential;
        private int precision;
        private int totalLength;

        private FloatFormat() {
        }

        @Override
        public void init(NDArray array) {
            Number[] values = array.toArray();
            int maxIntPartLen = 0;
            int maxFractionLen = 0;
            int expFractionLen = 0;
            int maxExpSize = 2;
            boolean sign = false;
            double max = 0.0;
            double min2 = Double.MAX_VALUE;
            for (Number n : values) {
                int fullFractionLen;
                int intPartLen;
                int expSize;
                double v = n.doubleValue();
                if (v < 0.0) {
                    sign = true;
                }
                if (!Double.isFinite(v)) {
                    int intPartLen2;
                    int n2 = intPartLen2 = v < 0.0 ? 4 : 3;
                    if (this.totalLength >= intPartLen2) continue;
                    this.totalLength = intPartLen2;
                    continue;
                }
                double abs = Math.abs(v);
                String str = String.format(Locale.ENGLISH, "%16e", abs);
                Matcher m4 = PATTERN.matcher(str);
                if (!m4.matches()) {
                    throw new AssertionError((Object)("Invalid decimal value: " + str));
                }
                int fractionLen = m4.group(1).length();
                if (expFractionLen < fractionLen) {
                    expFractionLen = fractionLen;
                }
                if ((expSize = m4.group(2).length()) > maxExpSize) {
                    maxExpSize = expSize;
                }
                if (abs >= 1.0) {
                    intPartLen = (int)Math.log10(abs) + 1;
                    if (v < 0.0) {
                        ++intPartLen;
                    }
                    if (intPartLen > maxIntPartLen) {
                        maxIntPartLen = intPartLen;
                    }
                    if (maxFractionLen < (fullFractionLen = fractionLen + 1 - intPartLen)) {
                        maxFractionLen = fullFractionLen;
                    }
                } else {
                    int n3 = intPartLen = v < 0.0 ? 2 : 1;
                    if (intPartLen > maxIntPartLen) {
                        maxIntPartLen = intPartLen;
                    }
                    if (maxFractionLen < (fullFractionLen = fractionLen + Integer.parseInt(m4.group(2)))) {
                        maxFractionLen = fullFractionLen;
                    }
                }
                if (abs > max) {
                    max = abs;
                }
                if (!(abs < min2) || !(abs > 0.0)) continue;
                min2 = abs;
            }
            double ratio = max / min2;
            if (max > 1.0E8 || min2 < 1.0E-4 || ratio > 1000.0) {
                this.exponential = true;
                this.precision = Math.min(8, expFractionLen);
                this.totalLength = this.precision + 4;
                if (sign) {
                    ++this.totalLength;
                }
            } else {
                this.precision = Math.min(4, maxFractionLen);
                int len = maxIntPartLen + this.precision + 1;
                if (this.totalLength < len) {
                    this.totalLength = len;
                }
            }
        }

        @Override
        public CharSequence format(Number value) {
            double d = value.doubleValue();
            if (Double.isNaN(d)) {
                return String.format(Locale.ENGLISH, "%" + this.totalLength + "s", "nan");
            }
            if (Double.isInfinite(d)) {
                if (d > 0.0) {
                    return String.format(Locale.ENGLISH, "%" + this.totalLength + "s", "inf");
                }
                return String.format(Locale.ENGLISH, "%" + this.totalLength + "s", "-inf");
            }
            if (this.exponential) {
                this.precision = Math.max(8, this.precision);
                return String.format(Locale.ENGLISH, "% ." + this.precision + "e", value.doubleValue());
            }
            if (this.precision == 0) {
                String fmt = "%" + (this.totalLength - 1) + '.' + this.precision + "f.";
                return String.format(Locale.ENGLISH, fmt, value.doubleValue());
            }
            String fmt = "%" + this.totalLength + '.' + this.precision + 'f';
            String ret = String.format(Locale.ENGLISH, fmt, value.doubleValue());
            char[] chars = ret.toCharArray();
            for (int i = chars.length - 1; i >= 0 && chars[i] == '0'; --i) {
                chars[i] = 32;
            }
            return new String(chars);
        }
    }
}

