/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http.policy;

import com.azure.core.http.HttpHeaderName;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpPipelineNextSyncPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.HttpLogDetailLevel;
import com.azure.core.http.policy.HttpLogOptions;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.http.policy.HttpRequestLogger;
import com.azure.core.http.policy.HttpRequestLoggingContext;
import com.azure.core.http.policy.HttpResponseLogger;
import com.azure.core.http.policy.HttpResponseLoggingContext;
import com.azure.core.implementation.AccessibleByteArrayOutputStream;
import com.azure.core.implementation.ImplUtils;
import com.azure.core.implementation.http.UrlSanitizer;
import com.azure.core.implementation.jackson.ObjectMapperShim;
import com.azure.core.implementation.util.BinaryDataContent;
import com.azure.core.implementation.util.BinaryDataHelper;
import com.azure.core.implementation.util.ByteArrayContent;
import com.azure.core.implementation.util.ByteBufferContent;
import com.azure.core.implementation.util.HttpHeadersAccessHelper;
import com.azure.core.implementation.util.InputStreamContent;
import com.azure.core.implementation.util.SerializableContent;
import com.azure.core.implementation.util.StringContent;
import com.azure.core.util.BinaryData;
import com.azure.core.util.Context;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.FluxUtil;
import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.logging.LogLevel;
import com.azure.core.util.logging.LoggingEventBuilder;
import com.fasterxml.jackson.databind.JsonNode;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class HttpLoggingPolicy
implements HttpPipelinePolicy {
    private static final ObjectMapperShim PRETTY_PRINTER = ObjectMapperShim.createPrettyPrintMapper();
    private static final int MAX_BODY_LOG_SIZE = 16384;
    private static final int LOGGER_CACHE_MAX_SIZE = 1000;
    private static final String CONTENT_LENGTH_KEY = HttpHeaderName.CONTENT_LENGTH.getCaseInsensitiveName();
    private static final Map<String, ClientLogger> CALLER_METHOD_LOGGER_CACHE = new ConcurrentHashMap<String, ClientLogger>();
    private static final ClientLogger LOGGER = new ClientLogger(HttpLoggingPolicy.class);
    private final HttpLogDetailLevel httpLogDetailLevel;
    private final Set<String> allowedHeaderNames;
    private final UrlSanitizer urlSanitizer;
    private final boolean prettyPrintBody;
    private final boolean disableRedactedHeaderLogging;
    private final HttpRequestLogger requestLogger;
    private final HttpResponseLogger responseLogger;
    public static final String RETRY_COUNT_CONTEXT = "requestRetryCount";
    private static final String REQUEST_LOG_MESSAGE = "HTTP request";
    private static final String RESPONSE_LOG_MESSAGE = "HTTP response";

    public HttpLoggingPolicy(HttpLogOptions httpLogOptions) {
        if (httpLogOptions == null) {
            this.httpLogDetailLevel = HttpLogDetailLevel.ENVIRONMENT_HTTP_LOG_DETAIL_LEVEL;
            this.allowedHeaderNames = HttpLogOptions.DEFAULT_HEADERS_ALLOWLIST.stream().map(HttpHeaderName::getCaseInsensitiveName).collect(Collectors.toSet());
            this.urlSanitizer = new UrlSanitizer(null);
            this.prettyPrintBody = false;
            this.disableRedactedHeaderLogging = false;
            this.requestLogger = new DefaultHttpRequestLogger();
            this.responseLogger = new DefaultHttpResponseLogger();
        } else {
            this.httpLogDetailLevel = httpLogOptions.getLogLevel();
            this.allowedHeaderNames = httpLogOptions.getAllowedHttpHeaderNames().stream().map(HttpHeaderName::getCaseInsensitiveName).collect(Collectors.toSet());
            this.urlSanitizer = new UrlSanitizer(httpLogOptions.getAllowedQueryParamNames());
            this.prettyPrintBody = httpLogOptions.isPrettyPrintBody();
            this.disableRedactedHeaderLogging = httpLogOptions.isRedactedHeaderLoggingDisabled();
            this.requestLogger = httpLogOptions.getRequestLogger() == null ? new DefaultHttpRequestLogger() : httpLogOptions.getRequestLogger();
            this.responseLogger = httpLogOptions.getResponseLogger() == null ? new DefaultHttpResponseLogger() : httpLogOptions.getResponseLogger();
        }
    }

    @Override
    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        if (this.httpLogDetailLevel == HttpLogDetailLevel.NONE) {
            return next.process();
        }
        ClientLogger logger2 = HttpLoggingPolicy.getOrCreateMethodLogger(context.getContext());
        long startNs = System.nanoTime();
        return this.requestLogger.logRequest(logger2, this.getRequestLoggingOptions(context)).then(next.process()).flatMap(response -> this.responseLogger.logResponse(logger2, this.getResponseLoggingOptions((HttpResponse)response, startNs, context))).doOnError(throwable -> this.createBasicLoggingContext(logger2, LogLevel.WARNING, context.getHttpRequest()).log("HTTP FAILED", throwable));
    }

    @Override
    public HttpResponse processSync(HttpPipelineCallContext context, HttpPipelineNextSyncPolicy next) {
        if (this.httpLogDetailLevel == HttpLogDetailLevel.NONE) {
            return next.processSync();
        }
        ClientLogger logger2 = HttpLoggingPolicy.getOrCreateMethodLogger(context.getContext());
        long startNs = System.nanoTime();
        this.requestLogger.logRequestSync(logger2, this.getRequestLoggingOptions(context));
        try {
            HttpResponse response = next.processSync();
            if (response != null) {
                response = this.responseLogger.logResponseSync(logger2, this.getResponseLoggingOptions(response, startNs, context));
            }
            return response;
        }
        catch (RuntimeException e) {
            this.createBasicLoggingContext(logger2, LogLevel.WARNING, context.getHttpRequest()).log("HTTP FAILED", e);
            throw e;
        }
    }

    private LoggingEventBuilder createBasicLoggingContext(ClientLogger logger2, LogLevel level, HttpRequest request) {
        LoggingEventBuilder log = logger2.atLevel(level);
        if (LOGGER.canLogAtLevel(level) && request != null) {
            String traceparent;
            String clientRequestId;
            if (this.allowedHeaderNames.contains(HttpHeaderName.X_MS_CLIENT_REQUEST_ID.getCaseInsensitiveName()) && (clientRequestId = request.getHeaders().getValue(HttpHeaderName.X_MS_CLIENT_REQUEST_ID)) != null) {
                log.addKeyValue(HttpHeaderName.X_MS_CLIENT_REQUEST_ID.getCaseInsensitiveName(), clientRequestId);
            }
            if (this.allowedHeaderNames.contains(HttpHeaderName.TRACEPARENT.getCaseInsensitiveName()) && (traceparent = request.getHeaders().getValue(HttpHeaderName.TRACEPARENT)) != null) {
                log.addKeyValue(HttpHeaderName.TRACEPARENT.getCaseInsensitiveName(), traceparent);
            }
        }
        return log;
    }

    private HttpRequestLoggingContext getRequestLoggingOptions(HttpPipelineCallContext callContext) {
        return new HttpRequestLoggingContext(callContext.getHttpRequest(), callContext.getContext(), HttpLoggingPolicy.getRequestRetryCount(callContext.getContext()));
    }

    private HttpResponseLoggingContext getResponseLoggingOptions(HttpResponse httpResponse, long startNs, HttpPipelineCallContext callContext) {
        return new HttpResponseLoggingContext(httpResponse, Duration.ofNanos(System.nanoTime() - startNs), callContext.getContext(), HttpLoggingPolicy.getRequestRetryCount(callContext.getContext()));
    }

    private void logBody(HttpRequest request, int contentLength, LoggingEventBuilder logBuilder, ClientLogger logger2, String contentType) {
        BinaryData data = request.getBodyAsBinaryData();
        BinaryDataContent content = BinaryDataHelper.getContent(data);
        if (content instanceof StringContent || content instanceof ByteBufferContent || content instanceof SerializableContent || content instanceof ByteArrayContent) {
            this.logBody(logBuilder, logger2, contentType, content.toString());
        } else if (content instanceof InputStreamContent) {
            byte[] contentBytes = content.toBytes();
            request.setBody(contentBytes);
            this.logBody(logBuilder, logger2, contentType, new String(contentBytes, StandardCharsets.UTF_8));
        } else {
            AccessibleByteArrayOutputStream stream = new AccessibleByteArrayOutputStream(contentLength);
            request.setBody(Flux.using(() -> stream, s2 -> content.toFluxByteBuffer().doOnNext(byteBuffer -> {
                try {
                    ImplUtils.writeByteBufferToStream(byteBuffer.duplicate(), s2);
                }
                catch (IOException ex) {
                    throw LOGGER.logExceptionAsError(new UncheckedIOException(ex));
                }
            }), s2 -> this.logBody(logBuilder, logger2, contentType, s2.toString(StandardCharsets.UTF_8))));
        }
    }

    private void logBody(LoggingEventBuilder logBuilder, ClientLogger logger2, String contentType, String data) {
        logBuilder.addKeyValue("body", HttpLoggingPolicy.prettyPrintIfNeeded(logger2, this.prettyPrintBody, contentType, data)).log(REQUEST_LOG_MESSAGE);
    }

    private static void addHeadersToLogMessage(Set<String> allowedHeaderNames, HttpHeaders headers, LoggingEventBuilder logBuilder, boolean disableRedactedHeaderLogging) {
        StringBuilder redactedHeaders = new StringBuilder();
        HttpHeadersAccessHelper.getRawHeaderMap(headers).forEach((key, value) -> {
            if (CONTENT_LENGTH_KEY.equals(key)) {
                return;
            }
            if (allowedHeaderNames.contains(key)) {
                logBuilder.addKeyValue(value.getName(), value.getValue());
            } else if (!disableRedactedHeaderLogging) {
                if (redactedHeaders.length() > 0) {
                    redactedHeaders.append(',');
                }
                redactedHeaders.append(value.getName());
            }
        });
        if (redactedHeaders.length() > 0) {
            logBuilder.addKeyValue("redactedHeaders", redactedHeaders.toString());
        }
    }

    private static String prettyPrintIfNeeded(ClientLogger logger2, boolean prettyPrintBody, String contentType, String body) {
        String result = body;
        if (prettyPrintBody && contentType != null && (contentType.startsWith("application/json") || contentType.startsWith("text/json"))) {
            try {
                JsonNode deserialized = PRETTY_PRINTER.readTree(body);
                result = PRETTY_PRINTER.writeValueAsString(deserialized);
            }
            catch (Exception e) {
                logger2.log(LogLevel.WARNING, () -> "Failed to pretty print JSON", e);
            }
        }
        return result;
    }

    private Long getAndLogContentLength(HttpHeaders headers, LoggingEventBuilder logBuilder, ClientLogger logger2) {
        String contentLengthString = headers.getValue(HttpHeaderName.CONTENT_LENGTH);
        if (CoreUtils.isNullOrEmpty(contentLengthString)) {
            return null;
        }
        try {
            Long contentLength = Long.parseLong(contentLengthString);
            logBuilder.addKeyValue(CONTENT_LENGTH_KEY, contentLength);
            return contentLength;
        }
        catch (NumberFormatException e) {
            logger2.atInfo().addKeyValue(CONTENT_LENGTH_KEY, contentLengthString).log("Could not parse the HTTP header content-length", e);
            return null;
        }
    }

    private static boolean shouldBodyBeLogged(String contentTypeHeader, Long contentLength) {
        return contentLength != null && !"application/octet-stream".equalsIgnoreCase(contentTypeHeader) && contentLength != 0L && contentLength < 16384L;
    }

    private static Integer getRequestRetryCount(Context context) {
        Object rawRetryCount = context.getData(RETRY_COUNT_CONTEXT).orElse(null);
        if (rawRetryCount == null) {
            return null;
        }
        try {
            return Integer.valueOf(rawRetryCount.toString());
        }
        catch (NumberFormatException ex) {
            LOGGER.atInfo().addKeyValue("tryCount", rawRetryCount).log("Could not parse the request retry count.");
            return null;
        }
    }

    private static ClientLogger getOrCreateMethodLogger(Context context) {
        ClientLogger logger2 = context.getData("caller-method-logger").orElse(null);
        if (logger2 != null) {
            return logger2;
        }
        String methodName = (String)context.getData("caller-method").orElse("");
        if (CALLER_METHOD_LOGGER_CACHE.size() > 1000) {
            CALLER_METHOD_LOGGER_CACHE.clear();
        }
        return CALLER_METHOD_LOGGER_CACHE.computeIfAbsent(methodName, ClientLogger::new);
    }

    private static LoggingEventBuilder getLogBuilder(LogLevel logLevel, ClientLogger logger2) {
        switch (logLevel) {
            case ERROR: {
                return logger2.atError();
            }
            case WARNING: {
                return logger2.atWarning();
            }
            case INFORMATIONAL: {
                return logger2.atInfo();
            }
        }
        return logger2.atVerbose();
    }

    private final class DefaultHttpRequestLogger
    implements HttpRequestLogger {
        private DefaultHttpRequestLogger() {
        }

        @Override
        public Mono<Void> logRequest(ClientLogger logger2, HttpRequestLoggingContext loggingOptions) {
            this.logRequestSync(logger2, loggingOptions);
            return Mono.empty();
        }

        @Override
        public void logRequestSync(ClientLogger logger2, HttpRequestLoggingContext loggingOptions) {
            this.log(this.getLogLevel(loggingOptions), logger2, loggingOptions);
        }

        private void log(LogLevel logLevel, ClientLogger logger2, HttpRequestLoggingContext loggingOptions) {
            if (!logger2.canLogAtLevel(logLevel) || HttpLoggingPolicy.this.httpLogDetailLevel == HttpLogDetailLevel.NONE) {
                return;
            }
            HttpRequest request = loggingOptions.getHttpRequest();
            LoggingEventBuilder logBuilder = HttpLoggingPolicy.getLogBuilder(logLevel, logger2).addKeyValue("method", (Object)request.getHttpMethod()).addKeyValue("url", HttpLoggingPolicy.this.urlSanitizer.getRedactedUrl(request.getUrl()));
            Integer retryCount = loggingOptions.getTryCount();
            if (retryCount != null) {
                logBuilder.addKeyValue("tryCount", retryCount);
            }
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogHeaders() && logger2.canLogAtLevel(LogLevel.INFORMATIONAL)) {
                HttpLoggingPolicy.addHeadersToLogMessage(HttpLoggingPolicy.this.allowedHeaderNames, request.getHeaders(), logBuilder, HttpLoggingPolicy.this.disableRedactedHeaderLogging);
            }
            Long contentLength = HttpLoggingPolicy.this.getAndLogContentLength(request.getHeaders(), logBuilder, logger2);
            if (request.getBody() == null) {
                logBuilder.log(HttpLoggingPolicy.REQUEST_LOG_MESSAGE);
                return;
            }
            String contentType = request.getHeaders().getValue(HttpHeaderName.CONTENT_TYPE);
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogBody() && HttpLoggingPolicy.shouldBodyBeLogged(contentType, contentLength)) {
                int contentLengthInt = contentLength.intValue();
                HttpLoggingPolicy.this.logBody(request, contentLengthInt, logBuilder, logger2, contentType);
                return;
            }
            logBuilder.log(HttpLoggingPolicy.REQUEST_LOG_MESSAGE);
        }
    }

    private final class DefaultHttpResponseLogger
    implements HttpResponseLogger {
        private DefaultHttpResponseLogger() {
        }

        @Override
        public Mono<HttpResponse> logResponse(ClientLogger logger2, HttpResponseLoggingContext loggingOptions) {
            String contentTypeHeader;
            LogLevel logLevel = this.getLogLevel(loggingOptions);
            HttpResponse response = loggingOptions.getHttpResponse();
            if (!logger2.canLogAtLevel(logLevel) || HttpLoggingPolicy.this.httpLogDetailLevel == HttpLogDetailLevel.NONE) {
                return Mono.just(response);
            }
            LoggingEventBuilder logBuilder = HttpLoggingPolicy.getLogBuilder(logLevel, logger2);
            this.addBasicResponseProperties(logger2, loggingOptions, response, logBuilder);
            Long contentLength = HttpLoggingPolicy.this.getAndLogContentLength(response.getHeaders(), logBuilder, logger2);
            Mono<HttpResponse> responseMono = Mono.just(response);
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogBody() && HttpLoggingPolicy.shouldBodyBeLogged(contentTypeHeader = response.getHeaderValue(HttpHeaderName.CONTENT_TYPE), contentLength)) {
                HttpResponse bufferedResponse = response.buffer();
                responseMono = FluxUtil.collectBytesInByteBufferStream(bufferedResponse.getBody()).map(bytes -> {
                    logBuilder.addKeyValue("body", HttpLoggingPolicy.prettyPrintIfNeeded(logger2, HttpLoggingPolicy.this.prettyPrintBody, contentTypeHeader, new String((byte[])bytes, StandardCharsets.UTF_8)));
                    return bufferedResponse;
                });
            }
            return responseMono.doOnNext(ignored -> logBuilder.log(HttpLoggingPolicy.RESPONSE_LOG_MESSAGE));
        }

        private void logHeaders(ClientLogger logger2, HttpResponse response, LoggingEventBuilder logBuilder) {
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogHeaders() && logger2.canLogAtLevel(LogLevel.INFORMATIONAL)) {
                HttpLoggingPolicy.addHeadersToLogMessage(HttpLoggingPolicy.this.allowedHeaderNames, response.getHeaders(), logBuilder, HttpLoggingPolicy.this.disableRedactedHeaderLogging);
            }
        }

        private void addBasicResponseProperties(ClientLogger logger2, HttpResponseLoggingContext loggingOptions, HttpResponse response, LoggingEventBuilder logBuilder) {
            logBuilder.addKeyValue("statusCode", response.getStatusCode()).addKeyValue("url", HttpLoggingPolicy.this.urlSanitizer.getRedactedUrl(response.getRequest().getUrl())).addKeyValue("durationMs", loggingOptions.getResponseDuration().toMillis());
            HttpLoggingPolicy.this.getAndLogContentLength(response.getHeaders(), logBuilder, logger2);
            this.logHeaders(logger2, response, logBuilder);
        }

        @Override
        public HttpResponse logResponseSync(ClientLogger logger2, HttpResponseLoggingContext loggingOptions) {
            String contentTypeHeader;
            LogLevel logLevel = this.getLogLevel(loggingOptions);
            HttpResponse response = loggingOptions.getHttpResponse();
            if (!logger2.canLogAtLevel(logLevel)) {
                return response;
            }
            LoggingEventBuilder logBuilder = HttpLoggingPolicy.getLogBuilder(logLevel, logger2);
            this.addBasicResponseProperties(logger2, loggingOptions, response, logBuilder);
            Long contentLength = HttpLoggingPolicy.this.getAndLogContentLength(response.getHeaders(), logBuilder, logger2);
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogBody() && HttpLoggingPolicy.shouldBodyBeLogged(contentTypeHeader = response.getHeaderValue(HttpHeaderName.CONTENT_TYPE), contentLength)) {
                response = response.buffer();
                logBuilder.addKeyValue("body", HttpLoggingPolicy.prettyPrintIfNeeded(logger2, HttpLoggingPolicy.this.prettyPrintBody, contentTypeHeader, response.getBodyAsBinaryData().toString()));
            }
            logBuilder.log(HttpLoggingPolicy.RESPONSE_LOG_MESSAGE);
            return response;
        }
    }
}

