/*
 * Decompiled with CFR 0.152.
 */
package com.nuix.automate.utils.logging;

import com.nuix.automate.utils.general.FormattingUtils;
import com.nuix.automate.utils.general.SchedulerComponent;
import com.nuix.automate.utils.logging.LogChannel;
import com.nuix.automate.utils.logging.LogHandler;
import com.nuix.automate.utils.logging.LogManagerUtils;
import com.nuix.automate.utils.logging.LoggerWrapper;
import com.nuix.automate.utils.models.internal.engine.EngineModel;
import com.nuix.automate.utils.models.internal.logging.LogEventModel;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

public class WorkerLogsWatchService {
    private static final LoggerWrapper LOGGER = LogManagerUtils.getLogger(WorkerLogsWatchService.class);
    private LogHandler logHandler = LogHandler.getInstance();
    private WatchService watchService;
    private Map<WatchKey, Path> keys;
    private Map<Path, LogChannel> channels = new HashMap<Path, LogChannel>();
    private Path logFolder;
    private String jobId;
    private String nuixJobGuid;
    private EngineModel engine;
    private Path workerDirectory;

    public WorkerLogsWatchService() {
        this.keys = new HashMap<WatchKey, Path>();
    }

    private void registerDirectory(Path dir) throws IOException {
        WatchKey key = dir.register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY);
        this.keys.put(key, dir);
        LOGGER.info("Registered " + dir.toString() + " worker directory ");
    }

    private String getRelativeParent(Path filePath) {
        return this.workerDirectory.relativize(filePath.getParent()).toString().replace("\\", "/");
    }

    private String getWorkerId(Path filePath) {
        if (filePath.getParent().equals(this.workerDirectory)) {
            return filePath.getFileName().toString();
        }
        return this.getWorkerId(filePath.getParent());
    }

    private void registerFile(Path filePath) throws IOException {
        SchedulerComponent component = this.componentFromFilename(filePath.getFileName().toString());
        if (component == null) {
            return;
        }
        File channel = filePath.toFile();
        String workerId = this.getWorkerId(filePath);
        String relativeParentPath = this.getRelativeParent(filePath);
        String fileName = filePath.getFileName().toString();
        long lastModified = filePath.toFile().lastModified();
        LogChannel workerChannel = new LogChannel("job-" + this.nuixJobGuid + "/" + FormattingUtils.sanitizeFilename(this.engine.getName()) + "/" + (relativeParentPath.length() == 0 ? "" : relativeParentPath + "/") + fileName, this.jobId, channel, component, lastModified);
        this.channels.put(filePath, workerChannel);
        if (relativeParentPath.length() == 0) {
            LOGGER.info("Registered " + component.name() + " from root job folder");
        } else if (!workerId.equals(relativeParentPath)) {
            LOGGER.info("Registered " + component.name() + " of worker " + workerId + " from sub folder " + relativeParentPath);
        } else {
            LOGGER.info("Registered " + component.name() + " of worker " + workerId);
        }
    }

    private void walkWorkerFolder(Path start) throws IOException {
        Files.walkFileTree(start, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                WorkerLogsWatchService.this.registerDirectory(dir);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                WorkerLogsWatchService.this.registerFile(file);
                if (WorkerLogsWatchService.this.channels.containsKey(file)) {
                    WorkerLogsWatchService.this.readChannelV2((LogChannel)WorkerLogsWatchService.this.channels.get(file));
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    private synchronized void processFileEvents() {
        if (this.channels.size() > 0) {
            for (Path filePath : this.channels.keySet()) {
                try {
                    if (this.channels.get(filePath) == null) continue;
                    long lastModified = filePath.toFile().lastModified();
                    if (this.channels.get(filePath).getLastModified() == lastModified) continue;
                    this.readChannelV2(this.channels.get(filePath));
                }
                catch (Exception e) {
                    LOGGER.info("Cannot read channel: " + e.getMessage());
                }
            }
        }
    }

    private synchronized void readChannelV2(LogChannel channel) throws IOException {
        if (channel == null) {
            return;
        }
        int bufferSize = (int)(channel.getChannel().length() - channel.getPosition());
        if (bufferSize > 0) {
            try (BufferedInputStream bin = new BufferedInputStream(Files.newInputStream(channel.getChannel().toPath(), new OpenOption[0]));){
                long skipped = bin.skip(channel.getPosition());
                byte[] buffer = new byte[bin.available()];
                int read = bin.read(buffer, 0, buffer.length);
                LogEventModel event = channel.getLogEventTemplate();
                event.setMessage(buffer);
                this.logHandler.addLogEvent(event);
                long position = skipped + (long)read;
                channel.setPosition(position);
            }
        } else if (channel.getPosition() < 1L) {
            LogEventModel event = channel.getLogEventTemplate();
            event.setMessage("".getBytes(StandardCharsets.UTF_8));
            this.logHandler.addLogEvent(event);
        }
    }

    public void processEvents() throws InterruptedException {
        while (true) {
            WatchKey key = this.watchService.poll(1L, TimeUnit.SECONDS);
            this.processFileEvents();
            Path dir = this.keys.get(key);
            if (dir == null) continue;
            for (WatchEvent<?> event : key.pollEvents()) {
                WatchEvent.Kind<?> kind = event.kind();
                Path name = (Path)event.context();
                Path child = dir.resolve(name);
                if (!kind.equals(StandardWatchEventKinds.ENTRY_CREATE)) continue;
                try {
                    this.walkWorkerFolder(child);
                }
                catch (IOException e) {
                    LOGGER.error("Failed to register " + child.toString() + " retrying");
                }
            }
            boolean valid = key.reset();
            if (valid) continue;
            this.keys.remove(key);
            if (this.keys.isEmpty()) break;
        }
    }

    public Thread getThread(Path logFolder, String jobId, String nuixJobGuid, EngineModel engine) {
        return new Thread(() -> {
            try {
                LOGGER.info("Starting watch service " + logFolder);
                this.watchService = FileSystems.getDefault().newWatchService();
                this.engine = engine;
                this.nuixJobGuid = nuixJobGuid;
                this.logFolder = logFolder;
                this.jobId = jobId;
                for (int count = 0; count != 300 && !logFolder.toFile().exists(); ++count) {
                    Thread.sleep(100L);
                }
                this.workerDirectory = Paths.get(logFolder.toString(), new String[0]);
                this.walkWorkerFolder(this.workerDirectory);
                this.processEvents();
            }
            catch (IOException e) {
                LOGGER.error("Unable to start watch service", e);
            }
            catch (InterruptedException e) {
                LOGGER.info("Closed watch service");
            }
            finally {
                try {
                    for (Path path : this.channels.keySet()) {
                        File channel = this.channels.get(path).getChannel();
                        if (channel == null) continue;
                        this.readChannelV2(this.channels.get(path));
                    }
                    this.watchService.close();
                }
                catch (IOException e2) {
                    LOGGER.error("Unable to close watch service", e2);
                }
            }
        });
    }

    private SchedulerComponent componentFromFilename(String filename) {
        switch (filename) {
            case "nuix.log": {
                return SchedulerComponent.NUIX_LOG;
            }
            case "derby.log": {
                return SchedulerComponent.NUIX_DERBY_LOG;
            }
            case "stdout.log": {
                return SchedulerComponent.NUIX_STD_OUT_LOG;
            }
            case "stderr.log": {
                return SchedulerComponent.NUIX_STD_ERR_LOG;
            }
        }
        return SchedulerComponent.OTHER;
    }
}

