Import-Module ExchangeOnlineManagement

function Has-Property([object] $Obj, [string] $Name) {
    [bool](Get-Member -InputObject $Obj -Name $Name -Type Properties)
}

function Copy-SameProperty([object] $From, [object] $To) {
    $From | Get-Member -Type Properties | % {if (Has-Property $To $_.Name) {$To.($_.Name) = $From.($_.Name)}}
}

class HoldPolicy {
    [string] $Id
    [string] $Name
    [string] $Comment
    [switch] $Enabled
    [string] $RuleName
    [string] $RuleComment
    [string] $ContentMatchQuery
    [string[]] $AddExchangeLocation
    [string[]] $AddPublicFolderLocation
    [string[]] $AddSharePointLocation
    [string[]] $RemoveExchangeLocation
    [string[]] $RemovePublicFolderLocation
    [string[]] $RemoveSharePointLocation

    HoldPolicy([object] $Obj) {
        Copy-SameProperty -From $Obj -To $this
    }
}

class EDiscoveryCase {
    [string] $UserServiceId,
    [string] $Id
    [string] $Name
    [string] $Description
    [string] $Type='eDiscovery'
    [swtich] $Enabled
    [string[]] $Managers
    [string[]] $Custodians
    [HoldPolicy[]] $HoldPolicies

    EDiscoveryCase([string] $Json) {
        $Obj = ConvertFrom-Json -inputObject $Json
        Copy-SameProperty -From $Obj -To $this
        $this.HoldPolicies = foreach ($_ in $Obj.HoldPolicies) {
            New-Object -TypeName HoldPolicy -ArgumentList $_
        }
    }
}

function Invoke-EDiscoverySession {
    Param(
        [Parameter(Mandatory)] $Username, 
        [Parameter(Mandatory)][Security.SecureString] $Password,
        [Parameter(Mandatory)][ScriptBlock] $ScriptBlock
    )

    $Credential = New-Object -TypeName System.Management.Automation.PSCredential -ArgumentList $Username, $Password
    Connect-IPPSSession -Credential $Credential

    try {
        Invoke-Command $ScriptBlock
    } finally {
        Disconnect-ExchangeOnline -Confirm:$false
    }
}

# Activate - Create case, hold policies and rules, and assign custodians and managers
function Create-EDiscoveryCase {
    param(
        [Parameter(Mandatory)][EDiscoveryCase] $Case,
        [string] $Username,
        [string] $Password,
        [switch] $UseExisting=$true
        )

    process {
        if (!('eDiscovery', 'AdvancedEDiscovery' -contains $Case.Type)) {
            $Case.Type = 'eDiscovery'
        }
        if ([string]::IsNullOrWhiteSpace($Case.Name)) {
            throw "Missing required case name"
        }

        # Start session and create case
        $Credentials = @{}
        if ($PSBoundParameters.ContainsKey('Username')) {$Credentials.Username = $Username}
        if ($PSBoundParameters.ContainsKey('Password')) {$Credentials.Password = $Password}

        Invoke-EDiscoverySession @Credentials -ScriptBlock {
            Write-Host "Creating new case: $($Case.Name)"
            $CaseParams = @{
                Name = $Case.Name
                CaseType = $Case.Type
            }
            if (![string]::IsNullOrWhiteSpace($Case.Description)) {$CaseParams.Description = $Case.Description}

            try {
                $case = New-ComplianceCase @CaseParams -ErrorAction Stop
            } catch {
                Write-Error $_
                if ($UseExisting) {
                    Write-Host "Error creating case, querying for existing case"
                    $case = Get-ComplianceCase -Identity $CaseParams.Name -ErrorAction Stop
                    Write-Host "Using existing case"

                    # # Update description
                    # if ($CaseParams.Description) {
                    #     Set-ComplianceCase -Case $case.Identity -Description $CaseParams.Description                
                    # }
                } else {
                    throw $_
                }
            }
            $Case.Id = $case.Identity

            # Create hold policies
            $Case.HoldPolicies | Create-HoldPolicyAndRule -Case $Case.Id
            $Case
        }
    }
}

function Create-HoldPolicyAndRule {
    param(
        [Parameter(Mandatory)][string] $Case,
        [Parameter(Mandatory)][HoldPolicy] $HoldPolicy
        )

    begin {
        Write-Host "Creating hold policies for case: $Case"
    }
    process {
        Write-Host "Creating new hold policy: $HoldPolicy.Name"
        $PolicyParams = @{
            Case = $Case
            Name = $HoldPolicy.Name
        }
        if (![string]::IsNullOrWhiteSpace($HoldPolicy.Comment)) {$PolicyParams.Comment = $HoldPolicy.Comment}
        if (![string]::IsNullOrWhiteSpace($HoldPolicy.Enabled)) {$PolicyParams.Enabled = $HoldPolicy.Enabled}
        if ($HoldPolicy.AddExchangeLocation.count -gt 0) {$PolicyParams.ExchangeLocation = $HoldPolicy.AddExchangeLocation}
        if ($HoldPolicy.AddPublicFolderLocation.count -gt 0) {$PolicyParams.PublicFolderLocation = $HoldPolicy.AddPublicFolderLocation}
        if ($HoldPolicy.AddSharePointLocation.count -gt 0) {$PolicyParams.SharePointLocation = $HoldPolicy.AddSharePointLocation}

        $HoldPolicyResult = New-HoldPolicy @PolicyParams
        $HoldPolicy.Id = $HoldPolicyResult.guid


        Write-Host "Creating new hold rule: $HoldPolicy.RuleName with query: $HoldPolicy.ContentMatchQuery"
        $RuleParams = @{
            Policy = $HoldPolicy.Id
            Name = $HoldPolicy.RuleName
        }
        if (![string]::IsNullOrWhiteSpace($HoldPolicy.RuleComment)) {$RuleParams.Comment = $HoldPolicy.RuleComment}
        if (![string]::IsNullOrWhiteSpace($HoldPolicy.ContentMatchQuery)) {$RuleParams.ContentMatchQuery = $HoldPolicy.ContentMatchQuery}

        New-CaseHoldRule @RuleParams
        $HoldPolicy
    }
}

function Update-EDiscoveryCase {
    param(
        [Parameter(Mandatory)][EDiscoveryCase] $Case,
        [string] $Username, 
        [Security.SecureString] $Passsword,
        [switch] $Close,
        [switch] $Reopen
        )

    process {
        $Identity = $Case.Id
        if (!$Identity) {
            $Identity = $Case.Name
        }
        if ([string]::IsNullOrWhiteSpace($Identity)) {
            throw "Missing required case identifier"
        }

        $CaseParams = @{
            Identity = $Identity
        }
        if (![string]::IsNullOrWhiteSpace($Case.Name)) {$CaseParams.Name = $Case.Name}
        if (![string]::IsNullOrWhiteSpace($Case.Type)) {$CaseParams.CaseType = $Case.Type}
        if (![string]::IsNullOrWhiteSpace($Case.Description)) {$CaseParams.Description = $Case.Description}
        if ($PSBoundParameters.ContainsKey('Close')) {$CaseParams.Close = $Close}
        if ($PSBoundParameters.ContainsKey('Reopen')) {$CaseParams.Reopen = $Reopen}

        # Start session and update case
        $Credentials = @{}
        if ($PSBoundParameters.ContainsKey('Username')) {$Credentials.Username = $Username}
        if ($PSBoundParameters.ContainsKey('Password')) {$CaseParams.Password = $Password}

        Invoke-EDiscoverySession @Credentials -ScriptBlock {
            Write-Host "Updating case: $Identity"
            Set-ComplianceCase @CaseParams

            # Update hold policies
            $Case.HoldPolicies | Update-HoldPolicyAndRule
            $Case
        }
    }
}

function Update-HoldPolicyAndRule {
    param(
        [Parameter(Mandatory)][HoldPolicy] $HoldPolicy
        )

    process {
        $PolicyIdentity = $HoldPolicy.Id
        if (!$PolicyIdentity) {
            $PolicyIdentity = $HoldPolicy.Name
        }
        if ([string]::IsNullOrWhiteSpace($PolicyIdentity)) {
            throw "Missing required hold policy identifier"
        }

        Write-Host "Updating hold policy: $PolicyIdentity"
        $PolicyParams = @{
            Identity = $PolicyIdentity
        }
        if (![string]::IsNullOrWhiteSpace($HoldPolicy.Comment)) {$PolicyParams.Comment = $HoldPolicy.Comment}
        if (![string]::IsNullOrWhiteSpace($HoldPolicy.Enabled)) {$PolicyParams.Enabled = $HoldPolicy.Enabled}

        $CurrentHoldPolicy = Get-HoldPolicy -Identity $PolicyIdentity

        $AddExchangeLocation = $HoldPolicy.ExchangeLocations | Where {$CurrentHoldPolicy.ExchangeLocation -NotContains $_}
        $RemoveExchangeLocation = $CurrentHoldPolicy.ExchangeLocation | Where {$HoldPolicy.ExchangeLocations -NotContains $_}

        $AddPublicFolderLocation = $HoldPolicy.PublicFolderLocations | Where {$CurrentHoldPolicy.PublicFolderLocation -NotContains $_}
        $RemovePublicFolderLocation = $CurrentHoldPolicy.PublicFolderLocation | Where {$HoldPolicy.PublicFolderLocations -NotContains $_}

        $AddSharePointLocation = $HoldPolicy.SharePointLocations | Where {$CurrentHoldPolicy.SharePointLocation -NotContains $_}
        $RemoveSharePointLocation = $CurrentHoldPolicy.SharePointLocation | Where {$HoldPolicy.SharePointLocations -NotContains $_}

        if ($AddExchangeLocation.count -gt 0) {$PolicyParams.AddExchangeLocation = $AddExchangeLocation}
        if ($RemoveExchangeLocation.count -gt 0) {$PolicyParams.RemoveExchangeLocation = $RemoveExchangeLocation}
        if ($AddPublicFolderLocation.count -gt 0) {$PolicyParams.AddPublicFolderLocation = $AddPublicFolderLocation}
        if ($RemovePublicFolderLocation.count -gt 0) {$PolicyParams.RemovePublicFolderLocation = $RemovePublicFolderLocation}
        if ($AddSharePointLocation.count -gt 0) {$PolicyParams.AddSharePointLocation = $AddSharePointLocation}
        if ($RemoveSharePointLocation.count -gt 0) {$PolicyParams.RemoveSharePointLocation = $RemoveSharePointLocation}

        Set-HoldPolicy @PolicyParams


        $RuleIdentity = $HoldPolicy.RuleName
        if ([string]::IsNullOrWhiteSpace($RuleIdentity)) {
            throw "Missing required hold rule identifier"
        }

        Write-Host "Updating hold rule: $RuleIdentity"
        $RuleParams = @{
            Identity = $RuleIdentity
        }
        if (![string]::IsNullOrWhiteSpace($HoldPolicy.RuleComment)) {$RuleParams.Comment = $HoldPolicy.RuleComment}
        if (![string]::IsNullOrWhiteSpace($HoldPolicy.ContentMatchQuery)) {$RuleParams.ContentMatchQuery = $HoldPolicy.ContentMatchQuery}

        Set-CaseHoldRule @RuleParams
    }
}

function Remove-EDiscoveryCase {
    param(
        [Parameter(Mandatory)][EDiscoveryCase] $Case,
        [string] $Username, 
        [Security.SecureString] $Passsword
        )

    process {
        $Identity = $Case.Id
        if (!$Identity) {
            $Identity = $Case.Name
        }
        if ([string]::IsNullOrWhiteSpace($Identity)) {
            throw "Missing required case identifier"
        }
        
        # Start session and create case
        $Credentials = @{}
        if ($PSBoundParameters.ContainsKey('Username')) {$Credentials.Username = $Username}
        if ($PSBoundParameters.ContainsKey('Password')) {$CaseParams.Password = $Password}

        Invoke-EDiscoverySession @Credentials -ScriptBlock {
            Write-Host "Removing case: $Identity"
            Write-Host "Removing case hold policies and rules"
            $Case.HoldPolicies | Remove-HoldPolicyAndRule

            Remove-ComplianceCase -Identity $Identity -Confirm:$false
        }
    }
}

function Remove-HoldPolicyAndRule {
    param(
        [Parameter(Mandatory, ParameterSetName='id')][string] $Id,
        [Parameter(Mandatory, ParameterSetName='name')][string] $Name
        )

    process {
        if ($PSCmdlet.ParameterSetName -eq 'id') {
            $Identity = $Id
        } elseif ($PSCmdlet.ParameterSetName -eq 'name') {
            $Identity = $Name
        }

        Write-Host "Removing hold policy and rule: $Identity"
        Get-CaseHoldRule -Policy $Identity | Remove-CaseHoldRule -Confirm:$false
        Remove-HoldPolicy -Identity $Identity -Confirm:$false
    }
}

function Add-CaseMember {
    param(
        [Parameter(Mandatory)][string] $Case,
        [Parameter(Mandatory)][string] $Member
        )

    begin {
        Write-Host "Adding members to case: $Case"
    }
    process {
        Write-Host "Adding member: $Member"
        Add-ComplianceCaseMember -Case $Case -Member $Member
    }
}

function Remove-CaseMember {
        param(
        [Parameter(Mandatory)][string] $Case,
        [Parameter(Mandatory)][string] $Member
        )

    begin {
        Write-Host "Removing members from case: $Case"
    }
    process {
        Write-Host "Removing member: $Member"
        Remove-ComplianceCaseMember -Case $Case -Member $Member
    }
}
